<?php

namespace HarikaTheme\TemplateBuilder;

defined('ABSPATH') || exit;

/**
 * Builder Frontend Class
 *
 * @since 2.0.0
 */
class Template_Frontend
{

    protected $is_header;
    protected $header_id;

    protected $is_footer;
    protected $footer_id;

    protected $is_post_single;
    protected $post_single_id;

    protected $is_post_cat;
    protected $post_cat_id;

    protected $is_megamenu;
    protected $megamenu_id;

    /**
     * Construct functions
     */
    public function __construct()
    {
        add_action('wp', function () {
            $this->init();

            //add_action( 'harika_builder_before_main', [$this, 'header'], 5 );

        });

        add_shortcode('harika-tb-block', [$this, 'blocks_shortcode']);

        add_shortcode('harika-tb-megamenu', [$this, 'megamenus_shortcode']);

        add_shortcode('harika-tb-post_single', [$this, 'post_singles_shortcode']);

        add_shortcode('harika-tb-post_cat', [$this, 'post_cats_shortcode']);

        add_shortcode('harika-tb-header', [$this, 'headers_shortcode']);

        add_shortcode('harika-tb-footer', [$this, 'footers_shortcode']);
    }

    /**
     * Get Frontend Template to display
     *
     * @return void
     */
    function init()
    {
        $this->get_settings('footer');
        //$this->get_settings( 'header' );
    }

    /**
     * Get frontend Templates
     *
     * @param string $type Template Type
     *
     * @return void
     */
    function get_settings($type)
    {
        $templates = $this->get_template_id($type);
        $template  = ! is_array($templates) ? $templates : $templates[0];

        if ('' !== $template) {
            switch ($type) {
                case 'footer':
                    $this->is_footer = true;
                    $this->footer_id = $template;
                    break;

                case 'header':
                    $this->is_header = true;
                    $this->header_id = $template;
                    break;

                case 'post_single':
                    $this->is_post_single = true;
                    $this->post_single_id = $template;
                    break;

                case 'post_cat':
                    $this->is_post_cat = true;
                    $this->post_cat_id = $template;
                    break;
            }
        }
    }

    /**
     * Get Item ID to display is any
     *
     * @param string $type Template Type
     *
     * @return void
     */
    public function get_template_id($type)
    {
        $templates = Template_Rule::instance()->get_templates_by_condition();

        foreach ($templates as $item) {
            if ($item['type'] === $type) {
                return $item['id'];
            }
        }

        return '';
    }

    /**
     * Hook Header template in page
     *
     * @return void
     */
    public function header()
    {
        if ($this->is_header) {
            $this->display('header');
        }
    }

    /**
     * Hook Footer template in page
     *
     * @return void
     */
    public function footer()
    {
        if ($this->is_footer) {
            $this->display('footer');
        }
    }


    /**
     * Display item
     *
     * @param string $type Template Type

     * @return void
     */
    public function display($type)
    {
        if ($type === 'header') {
            $id = $this->header_id;
            if ($id) {
?>
                <header class="site-header">
                    <?php echo self::get_elementor_content($id); ?>
                </header>
            <?php
            }
        }
        if ($type === 'footer') {
            $id = $this->footer_id;
            if ($id) {
            ?>
                <footer class="site-footer">
                    <?php echo self::get_elementor_content($id); ?>
                </footer>
        <?php
            }
        }


        if ($type === 'post_single') {
            $id      = $this->post_single_id;

            if ($id && ! \Elementor\Plugin::$instance->preview->is_preview_mode()) {
                $content = self::get_elementor_content($id);
                self::post_single_markup($content, $id);
            }
        }


        if ($type === 'post_cat') {
            $id      = $this->post_cat_id;

            if ($id && ! \Elementor\Plugin::$instance->preview->is_preview_mode()) {
                $content = self::get_elementor_content($id);
                self::post_cat_markup($content, $id);
            }
        }

        if ($type === 'megamenu') {
            $id      = $this->megamenu_id;

            if ($id && ! \Elementor\Plugin::$instance->preview->is_preview_mode()) {
                $content = self::get_elementor_content($id);
                self::megamenu_markup($content, $id);
            }
        }
    }

    /**
     * Get Elementor content for display
     *
     * @param int $content_id
     */
    public static function get_elementor_content($content_id)
    {
        $content = '';
        if (\class_exists('\Elementor\Plugin')) {
            $elementor_instance = \Elementor\Plugin::instance();
            $content            = $elementor_instance->frontend->get_builder_content_for_display($content_id, true);
        }
        return $content;
    }


    /**
     * megamenu Markup
     *
     * @param $content
     * @param $id
     *
     * @return void
     */
    public static function megamenu_markup($content, $id, $editing = false)
    {
        $wrapper_style   = '';

        if ($editing) {
            $wrapper_class = 'harika-megamenu-wrapper show-megamenu editing';
        } else {
            $wrapper_class = 'harika-megamenu-wrapper';
        }

        ?>
        <div class="<?php echo esc_attr($wrapper_class) ?>" style="<?php echo esc_attr($wrapper_style) ?>">
            <div class="megamenu-overly"></div>
            <div class="megamenu-container">
                <?php echo $content; ?>
            </div>
        </div>
    <?php
    }


    /**
     * post single Markup
     *
     * @param $content
     * @param $id
     *
     * @return void
     */
    public static function post_single_markup($content, $id, $editing = false)
    {
        $wrapper_style   = '';

        if ($editing) {
            $wrapper_class = 'harika-post_single-wrapper show-post_single editing';
        } else {
            $wrapper_class = 'harika-post_single-wrapper';
        }

    ?>
        <div class="<?php echo esc_attr($wrapper_class) ?>" style="<?php echo esc_attr($wrapper_style) ?>">
            <div class="post_single-overly"></div>
            <div class="post_single-container">
                <?php echo $content; ?>
            </div>
        </div>
<?php
    }


    public static function post_cat_markup($content, $id, $editing = false)
    {
        $wrapper_style   = '';

        if ($editing) {
            $wrapper_class = 'harika-post_cat-wrapper show-post_cat editing';
        } else {
            $wrapper_class = 'harika-post_cat-wrapper';
        }

    ?>
        <div class="<?php echo esc_attr($wrapper_class) ?>" style="<?php echo esc_attr($wrapper_style) ?>">
            <div class="post_cat-overly"></div>
            <div class="post_cat-container">
                <?php echo $content; ?>
            </div>
        </div>
<?php
    }

    public function blocks_shortcode($atts)
    {
        $attr = shortcode_atts(
            [
                'id' => false,
            ],
            $atts
        );

        if ($attr['id']) {
            return self::get_elementor_content($attr['id']);
        }
    }

    public function megamenus_shortcode($atts)
    {
        $attr = shortcode_atts(
            [
                'id' => false,
            ],
            $atts
        );

        if ($attr['id']) {
            return self::get_elementor_content($attr['id']);
        }
    }

    public function post_singles_shortcode($atts)
    {
        $attr = shortcode_atts(
            [
                'id' => false,
            ],
            $atts
        );

        if ($attr['id']) {
            return self::get_elementor_content($attr['id']);
        }
    }

    public function post_cats_shortcode($atts)
    {
        $attr = shortcode_atts(
            [
                'id' => false,
            ],
            $atts
        );

        if ($attr['id']) {
            return self::get_elementor_content($attr['id']);
        }
    }

    public function headers_shortcode($atts)
    {
        $attr = shortcode_atts(
            [
                'id' => false,
            ],
            $atts
        );

        if ($attr['id']) {
            return self::get_elementor_content($attr['id']);
        }
    }

    public function footers_shortcode($atts)
    {
        $attr = shortcode_atts(
            [
                'id' => false,
            ],
            $atts
        );

        if ($attr['id']) {
            return self::get_elementor_content($attr['id']);
        }
    }
}

new Template_Frontend();
