<?php
/**
 * Helper functions
 *
 */
use HarikaTheme\Classes\Harika_Helper as Helper;


defined( 'ABSPATH' ) || exit;

/**
 * Retrieve the name of the highest priority template file that exists.
 *
 * @param string|array $template_names Template file(s) to search for, in order.
 * @param string       $origin_path    Template file(s) origin path. (../harika-toolkit/elementor/widgets/)
 * @param string       $override_path  New template file(s) override path. (../harika)
 *
 * @return string The template filename if one is located.
 */
function st_get_locate_template( $template_names, $origin_path, $override_path ) {
    $files = [];
    $file  = '';

    foreach ( (array) $template_names as $template_name ) {
        if ( file_exists( get_stylesheet_directory() . $override_path . $template_name ) ) {
            $file = get_stylesheet_directory() . $override_path . $template_name;
        } elseif ( file_exists( get_template_directory() . $override_path . $template_name ) ) {
            $file = get_template_directory() . $override_path . $template_name;
        } elseif ( file_exists( realpath( __DIR__ . '/..' ) . $origin_path . $template_name ) ) {
            $file = realpath( __DIR__ . '/..' ) . $origin_path . $template_name;
        }
        array_push( $files, $file );
    }

    return $files;
}

/**
 * Get a list of Posts
 *
 * @param string $post_type
 * @return array
 */
function st_select_post( $post_type = 'post' ) {
    $args = [
        'post_type'   => $post_type,
        'numberposts' => -1,
        'orderby'     => 'title',
        'order'       => 'ASC',
    ];

    $query_query = get_posts( $args );

    $posts = [];
    if ( $query_query ) {
        foreach ( $query_query as $query ) {
            $posts[$query->ID] = $query->post_title;
        }
    }

    return $posts;
}

/**
 * Get All Category For Query
 *
 * @param string $category
 * @return array
 */
function st_select_category( $category = 'category' ) {
    $terms = get_terms( [
        'taxonomy'   => $category,
        'hide_empty' => true,
    ] );

    $options = [];

    if ( ! empty( $terms ) && ! is_wp_error( $terms ) ) {
        foreach ( $terms as $term ) {
            $options[$term->slug] = $term->name;
        }
    }

    return $options;
}

/**
 * Get a list of all Contact Form 7
 *
 * @since 1.0.0
 * @return array
 */
function st_select_cf7() {
    $forms_list = [];

    if ( function_exists( 'wpcf7' ) ) {
        $forms = get_posts( [
            'post_type'      => 'wpcf7_contact_form',
            'post_status'    => 'publish',
            'posts_per_page' => -1,
            'orderby'        => 'title',
            'order'          => 'ASC',
        ] );

        if ( ! empty( $forms ) ) {
            $forms_list = wp_list_pluck( $forms, 'post_title', 'ID' );
        } else {
            $forms_list[0] = esc_html__( 'فرم تماسی یافت نشد', 'harika' );
        }
    } else {
        $forms_list[0] = esc_html__( 'لطفا افزونه فرم تماس 7 را نصب و فعال کنید', 'harika' );
    }

    return $forms_list;
}

/**
 * Get a list of all block builder block
 *
 * @param string $type
 * @return array
 */
function st_select_builder_block( $type = 'block' ) {
    $items = [];
    $lists = ['0' => __( '--- انتخاب تمپلیت ---', 'harika' )];

    $args = [
        'post_type'      => 'harika_template',
        'posts_per_page' => -1,
    ];

    $posts = get_posts( $args );

    foreach ( $posts as $post ) {
        $meta = get_post_meta( $post->ID, 'harika_tb_settings', true );

        if ( is_array($meta) && isset($meta['template_type']) && $type == $meta['template_type'] ) {
            $items[$post->ID] = $post->post_title;
        }
    }

    $lists = $lists + $items;

    return $lists;
}

/**
 * Get a list of all header builder block
 *
 * @param string $type
 * @return array
 */
function st_select_builder_header( $type = 'header' ) {
    $items = [];
    $lists = ['0' => __( 'پیش فرض', 'harika' )];

    $args = [
        'post_type'      => 'harika_template',
        'posts_per_page' => -1,
    ];

    $posts = get_posts( $args );

    foreach ( $posts as $post ) {
        $meta = get_post_meta( $post->ID, 'harika_tb_settings', true );

        // Check if $meta is an array and 'template_type' exists
        if ( is_array($meta) && isset($meta['template_type']) && $type == $meta['template_type'] ) {
            $items[$post->ID] = $post->post_title;
        }
    }

    $lists = $lists + $items;

    return $lists;
}

function st_select_builder_single_template( $type = 'post_single' ) {
    $items = [];
    $lists = ['0' => __( 'پیش فرض', 'harika' )];

    $args = [
        'post_type'      => 'harika_template',
        'posts_per_page' => -1,
    ];

    $posts = get_posts( $args );

    foreach ( $posts as $post ) {
        $meta = get_post_meta( $post->ID, 'harika_tb_settings', true );

        // Check if $meta is an array and 'template_type' exists
        if ( is_array($meta) && isset($meta['template_type']) && $type == $meta['template_type'] ) {
            $items[$post->ID] = $post->post_title;
        }
    }

    $lists = $lists + $items;

    return $lists;
}

function st_select_builder_post_cat_template( $type = 'post_cat' ) {
    $items = [];
    $lists = ['0' => __( 'پیش فرض', 'harika' )];

    $args = [
        'post_type'      => 'harika_template',
        'posts_per_page' => -1,
    ];

    $posts = get_posts( $args );

    foreach ( $posts as $post ) {
        $meta = get_post_meta( $post->ID, 'harika_tb_settings', true );

        // Check if $meta is an array and 'template_type' exists
        if ( is_array($meta) && isset($meta['template_type']) && $type == $meta['template_type'] ) {
            $items[$post->ID] = $post->post_title;
        }
    }

    $lists = $lists + $items;

    return $lists;
}

function st_select_builder_megamenu( $type = 'megamenu' ) {
    $items = [];
    $lists = ['0' => __( 'انتخاب کنید', 'harika' )];

    $args = [
        'post_type'      => 'harika_template',
        'posts_per_page' => -1,
    ];

    $posts = get_posts( $args );

    foreach ( $posts as $post ) {
        $meta = get_post_meta( $post->ID, 'harika_tb_settings', true );

        if ( is_array($meta) && isset($meta['template_type']) && $type == $meta['template_type'] ) {
            $items[$post->ID] = $post->post_title;
        }
    }

    $lists = $lists + $items;

    return $lists;
}

/**
 * Get a list of all footer builder block
 *
 * @param string $type
 * @return array
 */
function st_select_builder_footer( $type = 'footer' ) {
    $items = [];
    $lists = ['0' => __( 'پیش فرض', 'harika' )];

    $args = [
        'post_type'      => 'harika_template',
        'posts_per_page' => -1,
    ];

    $posts = get_posts( $args );

    foreach ( $posts as $post ) {
        $meta = get_post_meta( $post->ID, 'harika_tb_settings', true );

        if ( is_array($meta) && isset($meta['template_type']) && $type == $meta['template_type'] ) {
            $items[$post->ID] = $post->post_title;
        }
    }

    $lists = $lists + $items;

    return $lists;
}



/**
 * Get list of Elementor Template
 *
 * @return array
 */
function st_select_elementor_template() {
    $args = [
        'post_type'   => 'elementor_library',
        'numberposts' => -1,
        'orderby'     => 'title',
        'order'       => 'ASC',
        'tax_query'   => [
            'relation' => 'OR',
            [
                'taxonomy' => 'elementor_library_type',
                'field'    => 'slug',
                'terms'    => 'section',
            ],
            [
                'taxonomy' => 'elementor_library_type',
                'field'    => 'slug',
                'terms'    => 'page',
            ],
        ],
    ];

    $query_query = get_posts( $args );

    $lists = ['0' => __( '--- انتخاب تمپلیت ---', 'harika' )];
    $items = [];

    if ( $query_query ) {
        foreach ( $query_query as $query ) {
            $items[$query->ID] = $query->post_title;
        }
    }

    $lists = $lists + $items;

    return $lists;
}

/**
 * Social Share links
 */
function st_post_share_links() {
    global $post;

    if ( ! isset( $post->ID ) ) {
        return;
    }

    $share_items = Helper::get_option( 'social_share_item', [] );

    if ( array_key_exists( 'enabled', $share_items ) ) {
        $share = $share_items['enabled'];
    } else {
        $share = [];
    }

    $html = '';

    if ( array_key_exists( 'twitter', $share ) ) {
        $html .= '<li>
            <a target="_blank" href="' . esc_url( 'https://twitter.com/intent/tweet?text=' . get_the_title() . '&amp;url=' . get_permalink() ) . '">
                <i class="fab fa-twitter"></i>
            </a>
        </li>';
    }

    if ( array_key_exists( 'facebook', $share ) ) {
        $html .= '<li>
            <a target="_blank" href="' . esc_url( 'https://www.facebook.com/share.php?u=' . get_permalink() ) . '">
                <i class="fab fa-facebook-f"></i>
            </a>
        </li>';
    }

    if ( array_key_exists( 'pinterest', $share ) ) {
        $img_url = wp_get_attachment_image_url( get_post_thumbnail_id( $post->ID ), 'full' );

        $html .= '<li>
            <a target="_blank" href="' . esc_url( 'https://pinterest.com/pin/create/button/?url=' . get_permalink() . '&media=' . $img_url ) . '">
                <i class="fab fa-pinterest-p"></i>
            </a>
        </li>';
    }

    if ( array_key_exists( 'linkedin', $share ) ) {
        $html .= '<li>
            <a target="_blank" href="' . esc_url( 'http://www.linkedin.com/shareArticle?mini=true&url=' . substr( urlencode( get_permalink() ), 0, 1024 ) ) . '&title=' . esc_attr( substr( urlencode( html_entity_decode( get_the_title() ) ), 0, 200 ) ) . '">
                <i class="fab fa-linkedin-in"></i>
            </a>
        </li>';
    }

    if ( array_key_exists( 'reddit', $share ) ) {
        $html .= '<li>
            <a target="_blank" href="' . esc_url( 'https://reddit.com/submit?url=<URL>&amp;title=' . get_the_title() . '&amp;url=' . get_permalink() ) . '">
                <i class="fab fa-reddit-alien" aria-hidden="true"></i>
            </a>
        </li>';
    }

    if ( array_key_exists( 'whatsapp', $share ) ) {
        $html .= '<li>
            <a target="_blank" href="' . esc_url( 'https://wa.me/?text=' . get_the_title() ) . '">
                <i class="fab fa-whatsapp" aria-hidden="true"></i>
            </a>
        </li>';
    }

    if ( array_key_exists( 'telegram', $share ) ) {
        $html .= '<li>
                <a target="_blank" href="' . esc_url( 'https://telegram.me/share/url?url=<URL>&amp;text=' . get_the_title() . '&amp;url=' . get_permalink() ) . '">
                    <i class="fab fa-telegram-plane" aria-hidden="true"></i>
                </a>
        </li>';
    }

    echo '<ul>' . $html . '</ul>';
}

/**
 * Set posts per page for custom post types and taxonomies
 */
if ( ! function_exists( 'st_cpt_per_page' ) ) {
    add_filter( 'pre_get_posts', 'st_cpt_per_page' );

    function st_cpt_per_page( $query ) {

        if ( $query->is_post_type_archive( 'harika_portfolio' ) && ! is_admin() && $query->is_main_query() ) {
            $post_per_page = Helper::get_option( 'portfolio_post_per_page', '9' );
            $query->set( 'posts_per_page', $post_per_page );
        }

        return $query;
    }
}

/**
 * Get a list of all the allowed html tags.
 *
 * @param string $level Allowed levels are basic and intermediate
 * @return array
 */
function st_get_allowed_html_tags( $level = 'basic' ) {
    $allowed_html = [
        'b'      => [
            'class' => [],
            'id'    => [],
            'style' => [],
        ],
        'i'      => [
            'class' => [],
            'id'    => [],
            'style' => [],
        ],
        'u'      => [
            'class' => [],
            'id'    => [],
            'style' => [],
        ],
        's'      => [
            'class' => [],
            'id'    => [],
            'style' => [],
        ],
        'br'     => [
            'class' => [],
            'id'    => [],
            'style' => [],
        ],
        'em'     => [
            'class' => [],
            'id'    => [],
            'style' => [],
        ],
        'del'    => [
            'class' => [],
            'id'    => [],
            'style' => [],
        ],
        'ins'    => [
            'class' => [],
            'id'    => [],
            'style' => [],
        ],
        'sub'    => [
            'class' => [],
            'id'    => [],
            'style' => [],
        ],
        'sup'    => [
            'class' => [],
            'id'    => [],
            'style' => [],
        ],
        'code'   => [
            'class' => [],
            'id'    => [],
            'style' => [],
        ],
        'mark'   => [
            'class' => [],
            'id'    => [],
            'style' => [],
        ],
        'small'  => [
            'class' => [],
            'id'    => [],
            'style' => [],
        ],
        'strike' => [
            'class' => [],
            'id'    => [],
            'style' => [],
        ],
        'abbr'   => [
            'title' => [],
            'class' => [],
            'id'    => [],
            'style' => [],
        ],
        'span'   => [
            'class' => [],
            'id'    => [],
            'style' => [],
        ],
        'strong' => [
            'class' => [],
            'id'    => [],
            'style' => [],
        ],
    ];

    return $allowed_html;
}

/**
 * Escaped title html tags
 *
 * @param string $tag input string of title tag
 * @return string $default default tag will be return during no matches
 */

function st_escape_tags( $tag, $default = 'span' ) {

    $supports = ['h1', 'h2', 'h3', 'h4', 'h5', 'h6', 'div', 'span', 'p'];

    if ( ! in_array( $tag, $supports, true ) ) {
        return $default;
    }

    return $tag;
}

/**
 * Strip all the tags except allowed html tags
 *
 * The name is based on inline editing toolbar name
 *
 * @param string $string
 * @return string
 */
function st_kses_basic( $string = '' ) {
    return wp_kses( $string, st_get_allowed_html_tags( 'basic' ) );
}









/* harika new 4.2 */




const MINIMUM_PHP_VERSION = '7.4';

function is_active_elementor()
{
    if (! did_action('elementor/loaded')) {
        return false;
    }

    return true;
}

function admin_notice_minimum_php_version()
{

    $message = sprintf(
        /* translators: 1: Plugin name 2: PHP 3: Required PHP version */
        esc_html__('قالب "%1$s" به "%2$s" نسخه %3$s نیاز دارد.', 'harika'),
        '<strong>' . esc_html__('هاریکا', 'harika') . '</strong>',
        '<strong>' . esc_html__('PHP', 'harika') . '</strong>',
        MINIMUM_PHP_VERSION
    );

    printf('<div class="notice notice-warning is-dismissible"><p>%1$s</p></div>', $message);
}

function is_compatible()
{
    // Check for required PHP version
    if (version_compare(PHP_VERSION, MINIMUM_PHP_VERSION, '<')) {
        add_action('admin_notices', 'admin_notice_minimum_php_version');
        return false;
    }

    return true;
}
add_action('init', 'is_compatible');


///*******************************************************///
///                   *sticky sidebar*                    ///
///*******************************************************///
function harika_in_widget_form($t, $return, $instance)
{
    $instance = wp_parse_args((array) $instance, array('sticky' => 'no'));
    if (!isset($instance['sticky']))
        $instance['sticky'] = null; ?>
    <p style="padding: 10px 0;border-top: 1px dashed;margin-top: 50px;">
        <label for="<?php echo $t->get_field_id('sticky'); ?>">چسبنده:</label>
        <select id="<?php echo $t->get_field_id('sticky'); ?>" name="<?php echo $t->get_field_name('sticky'); ?>">
            <option <?php selected($instance['sticky'], 'no'); ?> value="">خیر</option>
            <option <?php selected($instance['sticky'], 'sticky'); ?>value="sticky">بله</option>
        </select>
    </p>
<?php
    $retrun = null;
    return array($t, $return, $instance);
}
function harika_in_widget_form_update($instance, $new_instance, $old_instance)
{
    $instance['width'] = isset($new_instance['width']);
    $instance['sticky'] = $new_instance['sticky'];
    return $instance;
}
function harika_dynamic_sidebar_params($params)
{
    global $wp_registered_widgets;
    $widget_id = $params[0]['widget_id'];
    $widget_obj = $wp_registered_widgets[$widget_id];
    $widget_opt = get_option($widget_obj['callback'][0]->option_name);
    $widget_num = $widget_obj['params'][0]['number'];
    if (isset($widget_opt[$widget_num]['width'])) {
        if (isset($widget_opt[$widget_num]['sticky']))
            $sticky = $widget_opt[$widget_num]['sticky'];
        else
            $sticky = '';
        $params[0]['before_widget'] = preg_replace('/class="/', 'class="' . $sticky . ' ',  $params[0]['before_widget'], 1);
    }
    return $params;
}
//Add input fields(priority 5, 3 parameters)
add_action('in_widget_form', 'harika_in_widget_form', 5, 3);
//Callback function for options update (priorität 5, 3 parameters)
add_filter('widget_update_callback', 'harika_in_widget_form_update', 5, 3);
//add class names (default priority, one parameter)
add_filter('dynamic_sidebar_params', 'harika_dynamic_sidebar_params');











///*******************************************************///
///                   *post meta boxes*                   ///
///*******************************************************///

///*******************************************************///
///                   *post meta boxes*                   ///
///*******************************************************///
function custom_meta_box_post_formats($object)
{
    wp_nonce_field(basename(__FILE__), "meta-box-nonce");
?>
    <div class="harika-meta-options">
        <h3 class="heading"><i class="eicons eicon-document-file"></i>فرمت پست</h3>

        <div class="option">
            <span class="title">ویدیو از لینک مستقیم</span>
            <div class="control">
                <input type="text" name="harika-meta-direct-video" id="harika-meta-direct-video" placeholder="لینک ویدیو" value="<?php echo get_post_meta($object->ID, "harika-meta-direct-video", true); ?>" class="harika-input-field">
            </div>
        </div>

        <div class="option">
            <span class="title">ویدیو از آپارات</span>
            <div class="control">
                <textarea name="harika-meta-aparat-video" id="harika-meta-aparat-video" type="text" placeholder="آی فریم آپارات" class="harika-textarea-field"><?php echo get_post_meta($object->ID, "harika-meta-aparat-video", true);  ?></textarea>
            </div>
        </div>

        <div class="option">
            <span class="title">صوت از لینک مستقیم</span>
            <div class="control">
                <input type="text" name="harika-meta-direct-audio" id="harika-meta-direct-audio" placeholder="لینک صوت" value="<?php echo get_post_meta($object->ID, "harika-meta-direct-audio", true); ?>" class="harika-input-field">
            </div>
        </div>

    </div>
<?php
}


function add_custom_meta_box()
{
    add_meta_box("harika-meta-box-post-formats", "رسانه پست", "custom_meta_box_post_formats", "post", "advanced", "high", null);
}

add_action("add_meta_boxes", "add_custom_meta_box");


function save_custom_meta_box($post_id, $post, $update)
{

    if (!isset($_POST["meta-box-nonce"]) || !wp_verify_nonce($_POST["meta-box-nonce"], basename(__FILE__)))
        return $post_id;

    if (!current_user_can("edit_post", $post_id))
        return $post_id;

    if (defined("DOING_AUTOSAVE") && DOING_AUTOSAVE)
        return $post_id;


    if (isset($_POST["harika-meta-direct-video"])) {
        $harika_meta_direct_video_value = $_POST["harika-meta-direct-video"];
    }
    update_post_meta($post_id, "harika-meta-direct-video", $harika_meta_direct_video_value);

    if (isset($_POST["harika-meta-aparat-video"])) {
        $harika_meta_aparat_video_value = $_POST["harika-meta-aparat-video"];
    }
    update_post_meta($post_id, "harika-meta-aparat-video", $harika_meta_aparat_video_value);

    if (isset($_POST["harika-meta-direct-audio"])) {
        $harika_meta_direct_audio_value = $_POST["harika-meta-direct-audio"];
    }
    update_post_meta($post_id, "harika-meta-direct-audio", $harika_meta_direct_audio_value);
}
add_action("save_post", "save_custom_meta_box", 10, 3);














///*******************************************************///
///                     *CUSTOM SVGs*                     ///
///*******************************************************///
function post_is_audio_svg()
{
    return '<svg xmlns="http://www.w3.org/2000/svg" width="23" height="16" viewBox="0 0 23 16" fill="none">
	<line x1="22" y1="6.99998" x2="22" y2="8.99998" stroke="white" stroke-width="2" stroke-linecap="round"/>
	<line x1="19" y1="6.00001" x2="19" y2="10" stroke="white" stroke-width="2" stroke-linecap="round"/>
	<line x1="16" y1="3.00002" x2="16" y2="13" stroke="white" stroke-width="2" stroke-linecap="round"/>
	<line x1="13" y1="6.00001" x2="13" y2="10" stroke="white" stroke-width="2" stroke-linecap="round"/>
	<line x1="10" y1="1.00001" x2="10" y2="15" stroke="white" stroke-width="2" stroke-linecap="round"/>
	<line x1="7" y1="3.00002" x2="7" y2="13" stroke="white" stroke-width="2" stroke-linecap="round"/>
	<line x1="4" y1="6.00001" x2="4" y2="10" stroke="white" stroke-width="2" stroke-linecap="round"/>
	<line x1="1" y1="6.99998" x2="1" y2="8.99998" stroke="white" stroke-width="2" stroke-linecap="round"/>
	</svg>';
}
function post_is_video_svg()
{
    return '<svg xmlns="http://www.w3.org/2000/svg" width="15" height="16" viewBox="0 0 15 16" fill="none">
	<path d="M2.2917 14.6951L2.80769 13.8385L2.2917 14.6951C3.10808 15.1868 3.96709 15.0056 4.70317 14.7022C5.41337 14.4095 6.27489 13.8905 7.29085 13.2784L7.36215 13.2354L9.78523 11.7758L9.85486 11.7339C10.8723 11.1211 11.7334 10.6024 12.3302 10.1081C12.9535 9.59181 13.5 8.93142 13.5 8C13.5 7.06858 12.9535 6.4082 12.3302 5.89189C11.7334 5.39761 10.8723 4.87895 9.85486 4.26612C9.83173 4.25219 9.80852 4.23821 9.78522 4.22418L7.36215 2.76456C7.33829 2.75019 7.31452 2.73587 7.29084 2.7216C6.27488 2.10954 5.41337 1.59051 4.70317 1.2978C3.96709 0.994423 3.10808 0.813176 2.2917 1.30495C1.49124 1.78713 1.22431 2.62268 1.11093 3.41946C0.999906 4.19974 0.999948 5.23137 0.999998 6.46163C0.999999 6.4878 1 6.51405 1 6.54039L1 9.45962L0.999998 9.53837C0.999948 10.7686 0.999906 11.8003 1.11093 12.5805C1.22431 13.3773 1.49124 14.2129 2.2917 14.6951Z" stroke="white" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
	</svg>';
}
function video_title_link_svg()
{
    return '<svg xmlns="http://www.w3.org/2000/svg" width="18" height="25" viewBox="0 0 18 25" fill="none">
	<path d="M7.82059 9.0416C7.276 8.89686 6.71437 9.22127 6.57399 9.76288C4.69908 16.9967 5.23315 18.8366 9.12271 19.8704C13.1829 20.9494 14.5194 19.4544 16.6953 11.4002C18.8711 3.34597 18.4672 1.38844 14.4071 0.309378C11.9991 -0.330579 10.5491 -0.0651694 9.31392 1.93692C8.98386 2.47189 9.30837 3.14738 9.91916 3.30971L9.95222 3.3185C10.4206 3.44298 10.9 3.21477 11.1705 2.81554C11.2658 2.67486 11.3562 2.55877 11.4418 2.46306C11.7109 2.16204 11.9105 2.08048 12.1351 2.04535C12.4487 1.99629 12.9677 2.02338 13.8786 2.26548C14.7896 2.50758 15.2527 2.74151 15.4995 2.93951C15.6763 3.08131 15.8081 3.25094 15.8894 3.6451C15.987 4.11805 15.9984 4.87709 15.8082 6.11623C15.6207 7.33711 15.2613 8.88386 14.7231 10.8761C14.1849 12.8683 13.7165 14.3861 13.2634 15.5363C12.8035 16.7038 12.4109 17.3555 12.088 17.7167C11.8189 18.0177 11.6192 18.0993 11.3947 18.1344C11.0811 18.1834 10.5621 18.1564 9.65114 17.9143C8.7402 17.6722 8.27706 17.4382 8.03023 17.2402C7.85347 17.0984 7.7217 16.9288 7.64038 16.5346C7.5428 16.0617 7.53134 15.3026 7.7216 14.0635C7.87714 13.0505 8.15105 11.8132 8.54723 10.2833C8.68748 9.74167 8.36518 9.18633 7.82059 9.0416Z" fill="#2B39BE"/>
	<path d="M4.12137 22.7345C5.03232 22.9766 5.55134 23.0037 5.86491 22.9547C6.08946 22.9195 6.28909 22.838 6.55821 22.5369C6.64378 22.4412 6.73423 22.3251 6.82952 22.1845C7.09996 21.7852 7.57939 21.557 8.04778 21.6815L8.08084 21.6903C8.69163 21.8526 9.01614 22.5281 8.68608 23.0631C7.45086 25.0652 6.00089 25.3306 3.59294 24.6906C-0.467222 23.6116 -0.871087 21.654 1.30475 13.5998C3.48058 5.54556 4.81713 4.05058 8.87729 5.12964C12.7669 6.16337 13.3009 8.00329 11.426 15.2371C11.2856 15.7787 10.724 16.1031 10.1794 15.9584C9.63482 15.8137 9.31252 15.2583 9.45277 14.7167C9.84895 13.1868 10.1229 11.9495 10.2784 10.9365C10.4687 9.69736 10.4572 8.93832 10.3596 8.46536C10.2783 8.07121 10.1465 7.90157 9.96977 7.75978C9.72294 7.56178 9.2598 7.32784 8.34886 7.08574C7.43791 6.84364 6.91889 6.81655 6.60532 6.86562C6.38076 6.90075 6.18114 6.98231 5.91201 7.28333C5.58909 7.64453 5.19653 8.29619 4.73665 9.46368C4.28354 10.6139 3.81508 12.1317 3.27689 14.1239C2.7387 16.1161 2.37929 17.6629 2.19183 18.8838C2.00157 20.1229 2.01303 20.8819 2.11061 21.3549C2.19193 21.7491 2.3237 21.9187 2.50046 22.0605C2.74729 22.2585 3.21043 22.4924 4.12137 22.7345Z" fill="#2B39BE"/>
	</svg>';
}
function playlist_video_svg()
{
    return '<svg xmlns="http://www.w3.org/2000/svg" width="24" height="24" viewBox="0 0 24 24" fill="none">
	<path d="M1 12C1 14.4477 1.13246 16.3463 1.46153 17.827C1.78807 19.2963 2.29478 20.2921 3.00136 20.9986C3.70794 21.7052 4.70365 22.2119 6.17298 22.5385C7.65366 22.8675 9.55232 23 12 23C14.4477 23 16.3463 22.8675 17.827 22.5385C19.2963 22.2119 20.2921 21.7052 20.9986 20.9986C21.7052 20.2921 22.2119 19.2963 22.5385 17.827C22.8675 16.3463 23 14.4477 23 12C23 9.55232 22.8675 7.65366 22.5385 6.17298C22.2119 4.70365 21.7052 3.70794 20.9986 3.00136C20.2921 2.29478 19.2963 1.78807 17.827 1.46153C16.3463 1.13246 14.4477 1 12 1C9.55232 1 7.65366 1.13246 6.17298 1.46153C4.70365 1.78807 3.70794 2.29478 3.00136 3.00136C2.29478 3.70794 1.78807 4.70365 1.46153 6.17298C1.13246 7.65366 1 9.55232 1 12Z" stroke="white" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
	<path d="M12.6403 15.7758L14.2557 14.8027C14.277 14.7899 14.2982 14.7771 14.3194 14.7644C14.9825 14.365 15.5755 14.0079 15.993 13.6621C16.4368 13.2945 16.8936 12.7692 16.8936 12C16.8936 11.2308 16.4368 10.7055 15.993 10.3379C15.5755 9.99207 14.9825 9.63499 14.3194 9.23562C14.2982 9.22287 14.277 9.21009 14.2557 9.19725L12.6403 8.22418C12.6186 8.21106 12.5969 8.19799 12.5752 8.18496C11.9134 7.78619 11.3201 7.42868 10.8227 7.22369C10.2997 7.00815 9.59778 6.84042 8.91602 7.2511C8.25018 7.65219 8.05056 8.33908 7.97083 8.89935C7.89342 9.4434 7.89348 10.1518 7.89355 10.9545C7.89355 10.9785 7.89355 11.0027 7.89355 11.0269L7.89355 12.9731C7.89355 12.9973 7.89355 13.0215 7.89355 13.0455C7.89348 13.8482 7.89342 14.5566 7.97083 15.1006C8.05056 15.6609 8.25018 16.3478 8.91602 16.7489C9.59778 17.1596 10.2997 16.9918 10.8227 16.7763C11.32 16.5713 11.9134 16.2138 12.5752 15.815C12.5969 15.802 12.6185 15.7889 12.6403 15.7758Z" stroke="white" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"/>
	</svg>';
}






///*******************************************************///
///                     *dark mode*                   ///
///*******************************************************///
function wp_body_classes($classes)
{

    if (isset($_COOKIE['night_mode'])) {
        $night_mode_cookie = esc_attr($_COOKIE['night_mode']);
    }
    // Adds a class of hfeed to non-singular pages.
    if (isset($night_mode_cookie) &&  $night_mode_cookie == '1') {
        $classes[] = 'dark_mode';
    }
    return $classes;
}
add_filter('body_class', 'wp_body_classes');





///*******************************************************///
///                     *dark mode*                   ///
///*******************************************************///
function default_dark_skin($classes)
{
    $harika_default_skin  = HarikaTheme\Classes\Harika_Helper::get_option('harika_default_skin', 'light');

    if ($harika_default_skin == 'dark' && !isset($_COOKIE['night_mode'])) {
        $classes[] = 'default_dark_skin';
    }
    return $classes;
}
add_filter('body_class', 'default_dark_skin');





///*******************************************************///
///         *Make the attributes array be string*         ///
///*******************************************************///


if (! function_exists('harika_stringify_attributes')) {
    function harika_stringify_attributes($attributes)
    {
        $atts = array();
        foreach ($attributes as $name => $value) {
            $atts[] = $name . '="' . esc_attr($value) . '"';
        }
        return implode(' ', $atts);
    }
}













///*******************************************************///
///       * get page, post or... exclusive header id*     ///
///*******************************************************///
function harika_get_page_header_id()
{
    $present_header = 'default';

    $select_default_header_op    = Helper::get_option('select_default_header', '0');

    if ($select_default_header_op != '' && $select_default_header_op != 'default' && $select_default_header_op != '0') {
        $present_header = $select_default_header_op;
    }

    if (is_singular()) {
        $header_post_meta   = Helper::get_meta('harika_post_meta', 'select_post_header_meta', 'default');

        if ($header_post_meta != '' && $header_post_meta != 'default' && $header_post_meta != '0') {
            $present_header = $header_post_meta;
        }
    }

    if (is_page()) {
        $header_page_meta   = Helper::get_meta('harika_page_meta', 'select_page_header_meta', 'default');

        if ($header_page_meta != '' && $header_page_meta != 'default' && $header_page_meta != '0') {
            $present_header = $header_page_meta;
        }
    }

    if (is_archive()) {
        $header_post_cat_meta   = Helper::get_term_meta('select_post_cat_header_meta', 'default');

        if ($header_post_cat_meta != '' && $header_post_cat_meta != 'default' && $header_post_cat_meta != '0') {
            $present_header = $header_post_cat_meta;
        }
    }

    return $present_header;
}


///*******************************************************///
///       * get page, post or... exclusive footer id*     ///
///*******************************************************///
function harika_get_page_footer_id()
{
    $present_footer = 'default';

    $select_default_footer_op    = Helper::get_option('select_default_footer', '0');

    if ($select_default_footer_op != '' && $select_default_footer_op != 'default' && $select_default_footer_op != '0') {
        $present_footer = $select_default_footer_op;
    }

    if (is_singular()) {
        $footer_post_meta   = Helper::get_meta('harika_post_meta', 'select_post_footer_meta', 'default');

        if ($footer_post_meta != '' && $footer_post_meta != 'default' && $footer_post_meta != '0') {
            $present_footer = $footer_post_meta;
        }
    }

    if (is_page()) {
        $footer_page_meta   = Helper::get_meta('harika_page_meta', 'select_page_footer_meta', 'default');

        if ($footer_page_meta != '' && $footer_page_meta != 'default' && $footer_page_meta != '0') {
            $present_footer = $footer_page_meta;
        }
    }

    if (is_archive()) {
        $footer_post_cat_meta   = Helper::get_term_meta('select_post_cat_footer_meta');

        if ($footer_post_cat_meta != '' && $footer_post_cat_meta != 'default' && $footer_post_cat_meta != '0') {
            $present_footer = $footer_post_cat_meta;
        }
    }


    return $present_footer;
}




///*******************************************************///
///       * get page, post or... exclusive sidebar id*     ///
///*******************************************************///
function harika_get_page_sidebar_title_layout()
{
    $present_footer = '1';

    $select_default_footer_op    = Helper::get_option('select_sidebar_title_layout', '0');

    if ($select_default_footer_op != '' && $select_default_footer_op != 'default' && $select_default_footer_op != '0') {
        $present_footer = $select_default_footer_op;
    }

    if (is_singular()) {
        $footer_post_meta   = Helper::get_meta('harika_post_meta', 'select_post_sidebar_title_layout', 'default');

        if ($footer_post_meta != '' && $footer_post_meta != 'default' && $footer_post_meta != '0') {
            $present_footer = $footer_post_meta;
        }
    }

    if (is_page()) {
        $footer_page_meta   = Helper::get_meta('harika_page_meta', 'select_page_sidebar_title_layout', 'default');

        if ($footer_page_meta != '' && $footer_page_meta != 'default' && $footer_page_meta != '0') {
            $present_footer = $footer_page_meta;
        }
    }

    return 'sidebar_style_' . $present_footer;
}


///*******************************************************///
///       * get post exclusive single template id*     ///
///*******************************************************///
function harika_get_post_single_template_id()
{
    $present_header = 'default';

    $select_default_single_template_op    = Helper::get_option('select_default_single_template', '0');

    if ($select_default_single_template_op != '' && $select_default_single_template_op != 'default' && $select_default_single_template_op != '0') {
        $present_header = $select_default_single_template_op;
    }

    $select_post_single_template_meta   = Helper::get_meta('harika_post_meta', 'select_post_single_template_meta', 'default');

    if ($select_post_single_template_meta != '' && $select_post_single_template_meta != 'default' && $select_post_single_template_meta != '0') {
        $present_header = $select_post_single_template_meta;
    }

    return $present_header;
}


///*******************************************************///
///       * get post exclusive post cat template id*     ///
///*******************************************************///
function harika_get_post_cat_template_id()
{
    $present_header = 'default';

    $select_default_blog_archive_template_op    = Helper::get_option('select_default_blog_archive_template', '0');

    if ($select_default_blog_archive_template_op != '' && $select_default_blog_archive_template_op != 'default' && $select_default_blog_archive_template_op != '0') {
        $present_header = $select_default_blog_archive_template_op;
    }

    if (is_archive()) {
        $select_blog_archive_template_taxonomy      = Helper::get_term_meta('select_blog_archive_template_taxonomy');

        if ($select_blog_archive_template_taxonomy != '' && $select_blog_archive_template_taxonomy != 'default' && $select_blog_archive_template_taxonomy != '0') {
            $present_header = $select_blog_archive_template_taxonomy;
        }
    }

    return $present_header;
}




///*******************************************************///
///         * create elementor colors from meta *         ///
///*******************************************************///
function harika_create_elementor_colors_from_meta()
{
    /* metabox colors */
    $e_colors = '';
    $primary_meta    = '';
    $secondary_meta  = '';
    $background_meta = '';

    if (is_singular()) {
        $get_primary_meta_post    = Helper::get_meta('harika_post_meta', 'primary_post_color_meta', '');
        $get_secondary_meta_post  = Helper::get_meta('harika_post_meta', 'secondary_post_color_meta', '');
        $get_background_meta_post = Helper::get_meta('harika_post_meta', 'body_post_color_meta', '');
    }

    $get_primary_meta_page    = Helper::get_meta('harika_page_meta', 'primary_page_color_meta', '');
    $get_secondary_meta_page  = Helper::get_meta('harika_page_meta', 'secondary_page_color_meta', '');
    $get_background_meta_page = Helper::get_meta('harika_page_meta', 'body_page_color_meta', '');


    /* set from post */
    if (is_singular()) {
        if ($get_primary_meta_post != '') {
            $primary_meta = '
            --e-global-color-primary: ' . Helper::get_meta('harika_post_meta', 'primary_post_color_meta', '') . ';
            --harika-primary-color: ' . Helper::get_meta('harika_post_meta', 'primary_post_color_meta', '') . ';
            ';
        }
        if ($get_secondary_meta_post != '') {
            $secondary_meta = '
            --e-global-color-secondary: ' . Helper::get_meta('harika_post_meta', 'secondary_post_color_meta', '') . ';
            --harika-secondary-color: ' . Helper::get_meta('harika_post_meta', 'secondary_post_color_meta', '') . ';
            ';
        }
        if ($get_background_meta_post != '') {
            $background_meta = '
            --e-global-color-background: ' . Helper::get_meta('harika_post_meta', 'body_post_color_meta', '') . ';
            --harika-body-color: ' . Helper::get_meta('harika_post_meta', 'body_post_color_meta', '') . ';
            ';
        }
    }

    /* set from page */
    if (is_page()) {
        if ($get_primary_meta_page != '') {
            $primary_meta = '
            --e-global-color-primary: ' . Helper::get_meta('harika_page_meta', 'primary_page_color_meta', '') . ';
            --harika-primary-color: ' . Helper::get_meta('harika_page_meta', 'primary_page_color_meta', '') . ';
            ';
        }
        if ($get_secondary_meta_page != '') {
            $secondary_meta = '
            --e-global-color-secondary: ' . Helper::get_meta('harika_page_meta', 'secondary_page_color_meta', '') . ';
            --harika-secondary-color: ' . Helper::get_meta('harika_page_meta', 'secondary_page_color_meta', '') . ';
            ';
        }
        if ($get_background_meta_page != '') {
            $background_meta = '
            --e-global-color-background: ' . Helper::get_meta('harika_page_meta', 'body_page_color_meta', '') . ';
            --harika-body-color: ' . Helper::get_meta('harika_page_meta', 'body_page_color_meta', '') . ';
            ';
        }
    }

    if ($primary_meta != '' || $secondary_meta != '' || $background_meta != '') {
        $e_colors = '
        html body.elementor-kit-' . get_option('elementor_active_kit') . ' {
            ' . $primary_meta . '
            ' . $secondary_meta . '
            ' . $background_meta . '
        }
        ';
    }

    return $e_colors;
}






///*******************************************************///
///                     *excerpt length*                  ///
///*******************************************************///
function wpdocs_custom_excerpt_length($length)
{
    $excerpt_length = 32;

    // if(harika_get_setting( 'harika_archive_pages_excerpt_length' ) !== '') {
    // $excerpt_length = harika_get_setting( 'harika_archive_pages_excerpt_length' );
    // }

    return $excerpt_length;
}
add_filter('excerpt_length', 'wpdocs_custom_excerpt_length', 999);


///*******************************************************///
///                     *excerpt more*                    ///
///*******************************************************///
function wpdocs_excerpt_more($more)
{
    $more = sprintf(' ...');
    return $more;
}
add_filter('excerpt_more', 'wpdocs_excerpt_more');












///*******************************************************///
///                  *افزایش تعداد بازدیدها*                  ///
///*******************************************************///
function set_post_views($postID)
{
    $count_key = 'post_views_count';
    $count = get_post_meta($postID, $count_key, true);

    if ($count == '') {
        $count = 0;
        delete_post_meta($postID, $count_key);
        add_post_meta($postID, $count_key, '0');
    } else {
        $count++;
        update_post_meta($postID, $count_key, $count);
    }
}

///*******************************************************///
///               *جلوگیری از کش شدن بازدیدها*              ///
///*******************************************************///
function count_post_views($postID)
{
    if (!is_single()) return;
    if (empty($postID)) {
        global $post;
        $postID = $post->ID;
    }
    set_post_views($postID);
}
add_action('wp_head', 'count_post_views');



// جلوگیری از نمایش تعداد بازدید در خروجی
function get_post_views($postID)
{
    $count_key = 'post_views_count';
    $count = get_post_meta($postID, $count_key, true);
    if ($count == '') {
        delete_post_meta($postID, $count_key);
        add_post_meta($postID, $count_key, '0');
        return __('بدون بازدید', 'harika');
    }
    $result = $count . __(' بازدید', 'harika');
    if ($count == '0') {
        $result = __('بدون بازدید', 'harika');
    }
    return $result;
}


//////////////////////


// اضافه کردن ستون جدید فقط برای نوشته‌ها
function add_views_column($columns)
{
    $screen = get_current_screen();
    if ($screen->post_type === 'post') {
        $columns['post_views'] = __('تعداد بازدید', 'harika');
    }
    return $columns;
}
add_filter('manage_posts_columns', 'add_views_column');

// نمایش تعداد بازدید در ستون سفارشی فقط برای نوشته‌ها
function show_post_views_column($column_name, $post_id)
{
    if ($column_name === 'post_views' && get_post_type($post_id) === 'post') {
        $views = get_post_meta($post_id, 'post_views_count', true);
        echo '<div class="post-views-count">';
        echo $views ? $views : '0';
        echo '</div>';
    }
}
add_action('manage_posts_custom_column', 'show_post_views_column', 10, 2);

// تنظیم عرض ستون جدید فقط برای نوشته‌ها (اختیاری)
function set_custom_column_width()
{
    $screen = get_current_screen();
    if ($screen->post_type === 'post') {
        echo '<style>.column-post_views { width: 100px; }</style>';
    }
}
add_action('admin_head', 'set_custom_column_width');

// اضافه کردن قابلیت مرتب‌سازی به ستون "تعداد بازدید" فقط برای نوشته‌ها
function make_views_column_sortable($columns)
{
    $screen = get_current_screen();
    if ($screen->post_type === 'post') {
        $columns['post_views'] = 'post_views';
    }
    return $columns;
}
add_filter('manage_edit-post_sortable_columns', 'make_views_column_sortable');

// مرتب‌سازی ستون بر اساس تعداد بازدید فقط برای نوشته‌ها
function sort_views_column_query($query)
{
    if (!is_admin() || $query->get('post_type') !== 'post') return;

    $orderby = $query->get('orderby');
    if ('post_views' === $orderby) {
        $query->set('meta_key', 'post_views_count');
        $query->set('orderby', 'meta_value_num');
    }
}
add_action('pre_get_posts', 'sort_views_column_query');

// اضافه کردن آیکن به ستون تعداد بازدید فقط برای نوشته‌ها (اختیاری)
function add_views_column_icon()
{
    $screen = get_current_screen();
    if ($screen->post_type === 'post') {
        echo '<style>
            td.column-post_views .post-views-count:before {
                content: "\f177"; /* آیکن dashicon */
                font-family: dashicons;
            }
            .column-post_views .post-views-count {
                display: flex;
                align-items: center;
                gap: 5px;
                background: #2b39be;
                width: fit-content;
                padding: 0px 5px;
                color: #fff;
                border-radius: 5px;
            }
        </style>';
    }
}
add_action('admin_head', 'add_views_column_icon');
















///*******************************************************///
///           *check the elementor experiment*            ///
///*******************************************************///
function check_elementor_experiment_like_harika()
{

    $ex_disable_color_schemes =          get_option('elementor_disable_color_schemes'); // yes
    $ex_disable_typography_schemes =     get_option('elementor_disable_typography_schemes'); // yes
    $ex_e_font_icon_svg =                get_option('elementor_experiment-e_font_icon_svg'); // inactive
    $ex_inline_font_icons =              get_option('elementor_experiment-inline_font_icons'); // inactive
    $ex_container =                      get_option('elementor_experiment-container'); // active
    $ex_nested_elements =                get_option('elementor_experiment-nested-elements'); // active
    $ex_additional_custom_breakpoints =  get_option('elementor_experiment-additional_custom_breakpoints'); // active
    $ex_editor_v2 =                      get_option('elementor_experiment-editor_v2'); // active

    echo '<div class="codestar-custom-callback text">';
    if ($ex_disable_color_schemes == 'yes' and $ex_disable_typography_schemes == 'yes' and $ex_e_font_icon_svg == 'inactive' and $ex_inline_font_icons == 'inactive' and $ex_container == 'active' and $ex_nested_elements == 'active' and $ex_additional_custom_breakpoints == 'active' and $ex_editor_v2 == 'active') {
        echo 'تمامی ویژگی های المنتور مربوط به قالب هاریکا فعال هستند! ✅';
    } else {
        echo '❌ برخی از ویژگی های المنتور فعال یا غیرفعال هستند که باعث عدم سازگاری قالب با المنتور میشوند<br>
        لطفا تیک را بزنید و تنظیمات را مجددا ذخیره کنید تا ویژگی ها تنظیم شوند!';
    }
    echo '</div>';
}






///*******************************************************///
///      *Remove parsidate plugin ads from dashboard*     ///
///*******************************************************///
function wpp_remove_our_dashboard_primary_widget()
{
    remove_meta_box('wpp_dashboard_primary', 'dashboard', 'normal');
}
add_action('wp_dashboard_setup', 'wpp_remove_our_dashboard_primary_widget', 20);






///*******************************************************///
///         *print_r in a ltr way (just for dev)*         ///
///*******************************************************///
function print_h($input)
{
    echo '<pre style="direction: ltr; text-align:left;">';
    print_r($input);
    echo '</pre>';
}
